#!/bin/bash

# Variables
REPLICATION_SLOT="test_slot"
PUBLICATION_NAME="test_publication"
REPLICATION_USER="repuser"
RECEIVE_TIMEOUT="5s"

# Function to drop the replication slot
drop_replication_slot() {
        echo "Dropping replication slot: $REPLICATION_SLOT"
            psql -p 6972 postgres -c "SELECT pg_drop_replication_slot('$REPLICATION_SLOT');"
}

# Function to create a new logical replication slot with a specific restart_lsn
create_replication_slot() {
        psql -p 6972 postgres -c "SELECT pg_create_logical_replication_slot('$REPLICATION_SLOT', 'pgoutput', false);"
    echo "Advancing slot to current lsn"
    psql -p 6972 postgres -c "SELECT pg_replication_slot_advance('$REPLICATION_SLOT', pg_current_wal_lsn());"
}

# Ensure the replication slot does not already exist
psql -p 6972 postgres  -c "SELECT pg_drop_replication_slot('$REPLICATION_SLOT');" 2>/dev/null

# Create replication slot
psql -p 6972 postgres -c "SELECT * FROM pg_create_logical_replication_slot('$REPLICATION_SLOT', 'pgoutput');"

# Create tables
psql -p 6972 postgres <<EOF
CREATE TABLE published_table(id SERIAL PRIMARY KEY, value TEXT);
CREATE TABLE non_published_table(id SERIAL PRIMARY KEY, value TEXT);
EOF

# Create a publication for the published table
psql -p 6972 postgres -c "CREATE PUBLICATION $PUBLICATION_NAME FOR TABLE published_table;"

# Test logical replication performance when no transactions are part of the publication list
echo "### Testing performance when no transactions are part of the publication ###"

# Generate transactions on non-published table
psql -p 6972 postgres <<EOF
INSERT INTO non_published_table(value) SELECT md5(random()::text) FROM generate_series(1, 10000000);
EOF

# Start measuring time
start_time=$(date +%s%3N)

# Fetch the current WAL LSN as the end position for streaming
ENDPOS=$(psql -p 6972 -d postgres -t -c "SELECT pg_current_wal_lsn();")
echo "End LSN for published transactions: $ENDPOS"

# Capture the logical replication data
pg_recvlogical -p 6972 -d postgres -S $REPLICATION_SLOT --start --option=publication_names=test_publication --option=proto_version=3 -v --endpos $ENDPOS -f -   

# Stop measuring time
end_time=$(date +%s%3N)
duration_non_published=$((end_time - start_time))

# Drop and recreate the replication slot with the latest LSN
drop_replication_slot
create_replication_slot 


# Test logical replication performance when half of the transactions are part of the publication list
echo "### Testing performance when half of the transactions are part of the publication ###"

# Generate half of the transactions on the published table and other half on unpublished table
psql -p 6972 postgres <<EOF
INSERT INTO published_table(value) SELECT md5(random()::text) FROM generate_series(1, 5000000);INSERT INTO non_published_table(value) SELECT md5(random()::text) FROM generate_series(1, 5000000);
EOF
# Start measuring time
start_time=$(date +%s%3N)

# Fetch the current WAL LSN as the end position for streaming
ENDPOS=$(psql -p 6972 -d postgres -t -c "SELECT pg_current_wal_lsn();")
echo "End LSN for published transactions: $ENDPOS"

# Capture the logical replication data
pg_recvlogical -p 6972 -d postgres -S $REPLICATION_SLOT --start --option=publication_names=test_publication --option=proto_version=3 -v --endpos $ENDPOS -f /dev/null

# Stop measuring time
end_time=$(date +%s%3N)
duration_half_published=$((end_time - start_time))

# Drop and recreate the replication slot with the latest LSN
drop_replication_slot
create_replication_slot

# Test logical replication performance when all transactions are part of the publication list
echo "### Testing performance when all transactions are part of the publication ###"

# Generate transactions on the published table
psql -p 6972 postgres <<EOF
INSERT INTO published_table(value) SELECT md5(random()::text) FROM generate_series(1, 10000000);
EOF

# Start measuring time
start_time=$(date +%s%3N)

# Fetch the current WAL LSN as the end position for streaming
ENDPOS=$(psql -p 6972 -d postgres -t -c "SELECT pg_current_wal_lsn();")
echo "End LSN for published transactions: $ENDPOS"

# Capture the logical replication data
pg_recvlogical -p 6972 -d postgres -S $REPLICATION_SLOT --start --option=publication_names=test_publication --option=proto_version=3 -v --endpos $ENDPOS -f /dev/null    

# Stop measuring time
end_time=$(date +%s%3N)
duration_published=$((end_time - start_time))

# Print results
echo "Performance Results:"
echo " - Duration when no transactions are part of the publication: ${duration_non_published} ms"
echo " - Duration when half of the transactions are part of the publication: ${duration_half_published} ms"
echo " - Duration when all transactions are part of the publication: ${duration_published} ms"

# Clean up
psql -p 6972 postgres <<EOF
DROP PUBLICATION IF EXISTS $PUBLICATION_NAME;
DROP TABLE IF EXISTS published_table, non_published_table;
SELECT pg_drop_replication_slot('$REPLICATION_SLOT');
EOF
